/*
* Creation date : Tues Mar 03 09:00:00 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains implementation of 
* ECC signature function. 
*
* \version CE2_ECPKI_ECDSA.c#1:csrc:1
* \author Yermalayeu Ihar
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Include Files ***********************/

#include "CE2_ECPKI_ECDSA.h"
#include "LLF_ECPKI_ECDSA.h"
#include "LLF_ECPKI_Common.h"

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Global Data *************************/
/************************ Private function prototype **********/
/************************ Private Functions *******************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  CE2_DX_ECDSA_Sign
*
* Inputs:
* @param SignerPrivKey_ptr [in] - A pointer to a user private key structure.    				
* @param HashMode [in] - The enumerator variable defines hash function to be used.                         
* @param MessageDataIn_ptr [in] - A message data for calculation of hash.   			
* @param MessageSizeInBytes [in] - A size of block of message data in bytes. 
* @param SignatureOut_ptr [out] - A pointer to a buffer for output of signature.                         
* @param SignatureOutSize_ptr [out] - Size of user passed buffer for signature (in)
*                         and size of actual signature (out).
*
* Outputs:
* @returns \b CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_ECDSA_SIGN_INVALID_USER_PRIV_KEY_PTR_ERROR
*   - CE2_ECDSA_SIGN_ILLEGAL_HASH_OP_MODE_ERROR
*   - CE2_ECDSA_SIGN_INVALID_MESSAGE_DATA_IN_PTR_ERROR
*   - CE2_ECDSA_SIGN_INVALID_SIGNATURE_OUT_PTR_ERROR
*   - CE2_ECDSA_SIGN_INVALID_SIGNATURE_OUT_SIZE_PTR_ERROR
*
* \brief \b 
* Description:
*  Performs all of the ECDSA signing operations simultaneously.
*
* \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_ECDSA_Sign 
*     for ECDSA signing operations with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_DX_ECDSA_Sign(
									   CE2_ECPKI_UserPrivKey_t      *SignerPrivKey_ptr,       /*in*/
									   CE2_ECPKI_HASH_OpMode_t       HashMode,                /*in*/
									   DxUint8_t                     *MessageDataIn_ptr,       /*in*/ 
									   DxUint32_t                     MessageSizeInBytes,      /*in*/
									   DxUint8_t                     *SignatureOut_ptr,        /*out*/ 
									   DxUint32_t                    *SignatureOutSize_ptr,     /*out*/
									   DxInt8_t                      IsEphemerKeyInternal,    /*in*/
									   DxUint8_t                     *EphemerPublKeyIn_ptr,    /*in*/ 
									   DxUint32_t                     EphemerPublKeySizeBytes /*in*/)
{
	DxUint32_t orderSize;

	if (SignerPrivKey_ptr == DX_NULL)
		return CE2_ECDSA_SIGN_INVALID_USER_PRIV_KEY_PTR_ERROR;

	if (HashMode < 0 || HashMode >= CE2_ECPKI_HASH_NumOfModes)
		return CE2_ECDSA_SIGN_ILLEGAL_HASH_OP_MODE_ERROR;

	if ((MessageDataIn_ptr == DX_NULL) && (MessageSizeInBytes != 0))
		return CE2_ECDSA_SIGN_INVALID_MESSAGE_DATA_IN_PTR_ERROR;

	if (SignatureOut_ptr == DX_NULL)
		return CE2_ECDSA_SIGN_INVALID_SIGNATURE_OUT_PTR_ERROR;

	if (SignatureOutSize_ptr == DX_NULL)
		return CE2_ECDSA_SIGN_INVALID_SIGNATURE_OUT_SIZE_PTR_ERROR;

	if (IsEphemerKeyInternal != TRUE && EphemerPublKeyIn_ptr == DX_NULL)
		return CE2_ECDSA_SIGN_INVALID_EPHEMERAL_KEY_IN_ERROR;

	if (IsEphemerKeyInternal == TRUE && EphemerPublKeyIn_ptr != DX_NULL)
		return CE2_ECDSA_SIGN_INVALID_EPHEMERAL_KEY_IN_ERROR;

	orderSize = LLF_ECPKI_DomainIdToOrderSizeInBytes(SignerPrivKey_ptr->DomainID);
	if (*SignatureOutSize_ptr < 2*orderSize) {
		//    *SignatureOutSize_ptr = 2*orderSize;
		return CE2_ECDSA_SIGN_INVALID_SIGNATURE_OUT_SIZE_PTR_ERROR;
	}

	if (MessageSizeInBytes >= (1UL << 29))
		return CE2_ECDSA_SIGN_INVALID_MESSAGE_DATA_IN_PTR_ERROR;  

	return LLF_ECDSA_Sign(SignerPrivKey_ptr, HashMode, MessageDataIn_ptr,
		MessageSizeInBytes, SignatureOut_ptr, SignatureOutSize_ptr,IsEphemerKeyInternal,
		EphemerPublKeyIn_ptr, EphemerPublKeySizeBytes );
} /* End of CE2_ECDSA_Sign */

/**
****************************************************************
* Function Name: 
*  CE2_DX_ECDSA_Sign
*
* Inputs:
* @param SignerPrivKey_ptr [in] - A pointer to a user private key structure.    				
* @param HashMode [in] - The enumerator variable defines hash function to be used.                         
* @param MessageDataIn_ptr [in] - A message data for calculation of hash.   			
* @param MessageSizeInBytes [in] - A size of block of message data in bytes. 
* @param SignatureOut_ptr [out] - A pointer to a buffer for output of signature.                         
* @param SignatureOutSize_ptr [out] - Size of user passed buffer for signature (in)
*                         and size of actual signature (out).
*
* Outputs:
* @returns \b CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_ECDSA_SIGN_INVALID_USER_PRIV_KEY_PTR_ERROR
*   - CE2_ECDSA_SIGN_ILLEGAL_HASH_OP_MODE_ERROR
*   - CE2_ECDSA_SIGN_INVALID_MESSAGE_DATA_IN_PTR_ERROR
*   - CE2_ECDSA_SIGN_INVALID_SIGNATURE_OUT_PTR_ERROR
*   - CE2_ECDSA_SIGN_INVALID_SIGNATURE_OUT_SIZE_PTR_ERROR
*
* \brief \b 
* Description:
*  Performs all of the ECDSA signing operations simultaneously.
*
* \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function CE2_DX_ECDSA_Sign 
*     for ECDSA signing operations with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_ECDSA_Sign(CE2_ECPKI_UserPrivKey_t      *SignerPrivKey_ptr,       /*in*/
									   CE2_ECPKI_HASH_OpMode_t       HashMode,                /*in*/
									   DxUint8_t                     *MessageDataIn_ptr,       /*in*/ 
									   DxUint32_t                     MessageSizeInBytes,      /*in*/
									   DxUint8_t                     *SignatureOut_ptr,        /*out*/ 
									   DxUint32_t                    *SignatureOutSize_ptr     /*in*/)
{
	return CE2_DX_ECDSA_Sign(
		SignerPrivKey_ptr,
		HashMode, 
		MessageDataIn_ptr, 
		MessageSizeInBytes,
		SignatureOut_ptr,
		SignatureOutSize_ptr,
		TRUE, 
		DX_NULL,
		0 
		);
}
/**
****************************************************************
* Function Name: 
*  CE2_ECDSA_Verify
*
* Inputs:
* @param UserPublKey_ptr [in] - A pointer to a user public key structure.
* @param HashMode [in] - The enumerator variable defines the hash function to be used.
* @param MessageDataIn_ptr [in] - Message data for calculating hash.   			
* @param MessageSizeInBytes [in] - Size of block of message data in bytes.       
* @param SignatureIn_ptr [in] - A pointer to a buffer for output of signature. 
* @param SignatureSizeBytes [in] - Size of signature, in bytes
*
* Outputs:
* @returns \b CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_ECDSA_VERIFY_INVALID_SIGNER_PUBL_KEY_PTR_ERROR
*   - CE2_ECDSA_VERIFY_ILLEGAL_HASH_OP_MODE_ERROR
*   - CE2_ECDSA_VERIFY_INVALID_SIGNATURE_IN_PTR_ERROR
*   - CE2_ECDSA_VERIFY_INVALID_MESSAGE_DATA_IN_PTR_ERROR
*
* \brief \b 
* Description:
*  Performs all of the ECDSA verifying operations simultaneously.
*
* \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_ECDSA_Verify 
*     for ECDSA verifying operations with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_ECDSA_Verify (
 	CE2_ECPKI_UserPublKey_t       *UserPublKey_ptr,        /*in*/
  CE2_ECPKI_HASH_OpMode_t        HashMode,               /*in*/
  DxUint8_t                      *SignatureIn_ptr,        /*in*/
  DxUint32_t                      SignatureSizeBytes,     /*in*/
  DxUint8_t                      *MessageDataIn_ptr,      /*in*/ 
  DxUint32_t                      MessageSizeInBytes      /*in*/ )
{
  if (UserPublKey_ptr == DX_NULL)
    return CE2_ECDSA_VERIFY_INVALID_SIGNER_PUBL_KEY_PTR_ERROR;

  if (HashMode < 0 || HashMode >= CE2_ECPKI_HASH_NumOfModes) 
    return CE2_ECDSA_VERIFY_ILLEGAL_HASH_OP_MODE_ERROR;

  if (SignatureIn_ptr == DX_NULL)
    return CE2_ECDSA_VERIFY_INVALID_SIGNATURE_IN_PTR_ERROR;

  if ((MessageDataIn_ptr == DX_NULL) && (MessageSizeInBytes != 0))
    return CE2_ECDSA_SIGN_INVALID_MESSAGE_DATA_IN_PTR_ERROR;

  if (MessageSizeInBytes >= (1UL << 29))
    return CE2_ECDSA_SIGN_INVALID_MESSAGE_DATA_IN_PTR_ERROR;  

  return LLF_ECDSA_Verify(UserPublKey_ptr, HashMode, SignatureIn_ptr,
    SignatureSizeBytes, MessageDataIn_ptr, MessageSizeInBytes);
} /* End of CE2_ECDSA_Verify */



